#pragma once
#include "Engine/ECS/Entity.h"
#include "Engine/ECS/TransformComponent.h"
#include "Engine/ECS/ImageComponent.h"
#include "Engine/ECS/CollisionComponent.h"
#include "Engine/ECS/InputComponent.h"
#include "Engine/ECS/PhysicsComponent.h"
#include "Breakout/Ball.h"
#include "Breakout/BreakoutScene.h"
#include "Config.h"

class Paddle : public Entity {
public:
  Paddle(BreakoutScene& Scene) : Entity{Scene} {
    Transform = AddComponent<TransformComponent>();
    Transform->SetPosition({
      5.0f * Scene.PIXELS_PER_METER,
      7.2f * Scene.PIXELS_PER_METER
    });

    Input = AddComponent<InputComponent>();
    Input->UnbindKey(SDLK_SPACE);
    Input->BindKeyHeld(SDLK_LEFT, CreateMoveLeftCommand);
    Input->BindKeyHeld(SDLK_RIGHT, CreateMoveRightCommand);

    float CollisionWidth{3.1f * Scene.PIXELS_PER_METER};
    float CollisionOffsetX{1.f * Scene.PIXELS_PER_METER};
    Physics = AddComponent<PhysicsComponent>();
    Physics->SetGravity({0, 0});
    Physics->ConstrainHorizontalMovement(
      -CollisionOffsetX,
      Scene.GetWidth() - (
        CollisionOffsetX + CollisionWidth
      )
    );

    Collision = AddComponent<CollisionComponent>();
    Collision->SetSize(
      CollisionWidth,
      0.6f * Scene.PIXELS_PER_METER
    );
    Collision->SetOffset({CollisionOffsetX, 0});

    AddComponent<ImageComponent>(
      Config::BASE_PATH + "Assets/Paddle_Frame_B.png"
    );

    SetIsPaused(true);
  }

  void Tick(float DeltaTime) override {
    Entity::Tick(DeltaTime);
    Physics->SetVelocity({0, 0});
  }

  void HandleEvent(const SDL_Event& E) override {
    if (
      E.type == SDL_EVENT_KEY_DOWN &&
      E.key.key == SDLK_SPACE &&
      GetScene().GetState() == GameState::InProgress
    ) {
      SetIsPaused(false);
    } else if (
      E.type == UserEvents::GAME_WON ||
      E.type == UserEvents::GAME_LOST
    ) {
      SetIsPaused(true);
    }
  }

  void HandleCollision(Entity& Other) override {
    if (Ball* Ptr{dynamic_cast<Ball*>(&Other)}) {
      HandleBallCollision(Ptr);

      Collision->SetIsEnabled(false);

      if (TimerID != 0) {
        SDL_RemoveTimer(TimerID);
      }

      TimerID = SDL_AddTimer(
        500, &Paddle::EnableCollision, this
      );
    }
  }

  Paddle& operator=(const Paddle& Other) = delete;
  Paddle(const Paddle& Other) = delete;

  ~Paddle() {
    if (TimerID) {
      SDL_RemoveTimer(TimerID);
    }
  }

private:
  void SetIsPaused(bool isPaused) {
    Input->SetIsEnabled(!isPaused);
    Physics->SetIsEnabled(!isPaused);
  }

  SDL_TimerID TimerID{0};
  TransformComponent* Transform{nullptr};
  CollisionComponent* Collision{nullptr};
  InputComponent* Input{nullptr};
  PhysicsComponent* Physics{nullptr};

  void HandleBallCollision(Ball* BallPtr) {
    Vec2 PaddlePos{Collision->GetCenter()};
    float PaddleWidth{Collision->GetSize().x};

    CollisionComponent* BallCollision{
      BallPtr->GetComponent<CollisionComponent>()
    };

    Vec2 BallPos{BallCollision->GetCenter()};
    float BallWidth{BallCollision->GetSize().x};

    // Where on the paddle the ball hit from
    // -1.0 (left edge) to 1.0 (right edge)
    float HitOffset{
      (BallPos.x - PaddlePos.x) / (
        (PaddleWidth + BallWidth) / 2
      )
    };

    Vec2 Direction{HitOffset, -1.0f};

    PhysicsComponent* BallPhysics{
      BallPtr->GetComponent<PhysicsComponent>()
    };

    BallPhysics->SetVelocity(
      Direction.Normalize() *
      Config::Breakout::BALL_SPEED *
      float(Scene::PIXELS_PER_METER)
    );
  }

  static Uint32 EnableCollision(
    void* Entity, SDL_TimerID, Uint32
  ) {
    Paddle* Target{static_cast<Paddle*>(Entity)};
    if (Target && Target->Collision) {
      Target->Collision->SetIsEnabled(true);
    }

    return 0;
  }

  static CommandPtr CreateMoveLeftCommand() {
    using namespace Config::Breakout;
    return std::make_unique<MovementCommand>(
      Vec2{
        -PADDLE_SPEED * Scene::PIXELS_PER_METER,
        0.0
      }
    );
  }

  static CommandPtr CreateMoveRightCommand() {
    using namespace Config::Breakout;
    return std::make_unique<MovementCommand>(
      Vec2{
        PADDLE_SPEED * Scene::PIXELS_PER_METER,
        0.0
      }
    );
  }
};